/*
 * Copyright 2019 NXP
 * All rights reserved.
 * SPDX-License-Identifier: BSD-3-Clause
 */

#ifndef _GTOF_H_
#define _GTOF_H_


///////////////////////////////////////////////////////////////////////////////////////////////////
//                                      Includes Section
///////////////////////////////////////////////////////////////////////////////////////////////////

#include <stdint.h>
#include "fsl_common.h"
#include "gtof_utils.h"
#include "genfsk_interface.h"

/*!
 * @addtogroup GENFSK_Localization GENFSK localization features
 * This module handles localization technics based on Generic FSK hardware module.
 * @{
 * @}
 */

/*!
 * @addtogroup TimeOfFlight Time of Flight localization feature
 * @ingroup GENFSK_Localization
 * Distance estimation using Time of Flight (ToF)
 * ToF consists in measurement over-the-air round trip time between two devices (Anchor and Reflector).
 * The Anchor (also called Measuring Device) is reposonsible for trigering the measurement protocol, collecting data and performing distance calculation.
 * The Reflector (also called Receiving Device) listen for Anchor requests and replies.
 * See "Localization Quick Start Guide" document for explanation on how to operate with ToF protocol and API.
 * @{
 */

/*! @file
 * GENFSK Localization - Time of Flight API
 */
///////////////////////////////////////////////////////////////////////////////////////////////////
//                                  Defines & Macros Section
///////////////////////////////////////////////////////////////////////////////////////////////////

/*! GFSKTOF Version */
#define GFSKTOF_VERSION (MAKE_VERSION(2, 1, 0))

/*! Customization of GENFSK header used by TOF and measuring applications */
#define GFSKTOF_CRCSIZE                          (3U)
#define GFSKTOF_LENGHT_FIELD_SIZE                (6U)
#define GFSKTOF_SYNC_ADDRESS_SIZE                (3U)
#define GFSKTOF_H0_BIT_SIZE                      (8U)
#define GFSKTOF_H1_BIT_SIZE                      (2U)
#define GFSKTOF_H0_VALUE                         (0U)
#define GFSKTOF_H0_MASK                          ((1U<<GFSKTOF_H0_BIT_SIZE)-1U)
#define GFSKTOF_H1_VALUE                         (0U)
#define GFSKTOF_H1_MASK                          ((1U<<GFSKTOF_H1_BIT_SIZE)-1U)

#define GFSKTOF_HEADER_SIZE                      ((GFSKTOF_H0_BIT_SIZE + GFSKTOF_LENGHT_FIELD_SIZE + GFSKTOF_H1_BIT_SIZE) >> 3U)

/* Compiler flags to customize ToF memory footprint */

#ifndef TOF_RSSI_LOGGING
#define TOF_RSSI_LOGGING                                (0)
#endif

#ifndef TOF_CHANNEL_LOGGING
#define TOF_CHANNEL_LOGGING                             (0)
#endif

///////////////////////////////////////////////////////////////////////////////////////////////////
//                                      Typedef Section
///////////////////////////////////////////////////////////////////////////////////////////////////

/*! Type of AGC control perform during ToF measurement */
typedef enum {
    GTOF_AGC_BEHAVIOR_UNLOCKED,     /*!< AGC is not locked and opearates normally */
    GTOF_AGC_BEHAVIOR_LOCKED_AUTO,  /*!< AGC is evaluated and locked to a single value during whole measurement */
    GTOF_AGC_BEHAVIOR_LOCKED_INDEX, /*!< AGC is locked, using the index provided separatly */
} gtofAgcBehavior_t;    

/*! @brief ToF completion event.
 * Event returned by ToF callback to notify about a ToF event.
 * @see tofCallback_t
 */
typedef enum
{
    TofMeasurementDoneEvent = 0x10,   /*!< Measurement success */
    TofCalibrationDoneEvent,          /*!< Calibration success */
    TofMeasurementErrorEvent,         /*!< An error occured during measurement, status available in gtofMeasureResults_t */
    TofCalibrationErrorEvent,         /*!< An error occured during calibration, status available in gtofMeasureResults_t */
} tofEvents_t;

/*! @brief ToF measurement status
 * Measurement status contained in a measurement result.
 * @see gtofMeasureResults_t
 */
typedef enum
{
    TofStatusOk,                       /*!< Success */
    TofStatusWrongOpcode,              /*!< A packet with unknown OP code has been received, measure aborted */
    TofStatusStartAckTimeout,          /*!< Start sequence timed-out despite retries */
    TofStatusCalibrationAckTimeout,    /*!< CalibrationAck wasn't received on time */
    TofStatusCalibrationResultInvalid, /*!< Calibration completed with an invalid value (out of bounds) */
    TofStatusInvalidParameter,
} tofStatus_t;

/*! @brief ToF data provided at the time of measurement completion.
 * @see tofCallback_t
 */
typedef struct
{
    fixedq6_t estimDistance;                   /*!< Estimated distance in meters (signed Q6 notation) */
    fixedq6_t estimTof;                        /*!< Estimated time of flight in ns (signed Q6 notation) */
    fixedq6_t calibrationBias;                 /*!< Calibration bias returned by calibration process */
    uint16_t nbSamples;                        /*!< Number of samples actually captured during the measurement */
    uint8_t nbRetries;                         /*!< Number of packet retries that happened during the measurement */
    uint8_t nbHops;                            /*!< Number of hops that happened during the measurement */
    GENFSK_timestamp_t measurementStartTime;   /*!< Timestamp at the beginning of measurement */
    GENFSK_timestamp_t measurementEndTime;     /*!< Timestamp at the end of measurement */
    tofStatus_t tofStatus;                     /*!< execution status of measurement */
} gtofMeasureResults_t;

/*! @brief Application callback to catch ToF events.
 * @param event The type of notification raised by ToF API
 * @param eventData Measurement Results
 *
 * @see tofEvents_t
 */
typedef void (* tofCallback_t)(tofEvents_t event, gtofMeasureResults_t *eventData);

/*! @brief Parameters for GTOF/application interaction */
typedef struct
{
    uint8_t GenfskInstanceId;                               /*!< GENFSK instanceId initialized by application, to be used for ToF measurement*/
    genfskPacketReceivedCallBack_t PacketReceivedCallback;  /*!< Callback to handle packets from GENFSK while ToF is not running but initialised */
    genfskEventNotifyCallBack_t EventCallback;              /*!< Callback to handle events from GENFSK while ToF is not running but initialised */
    tofCallback_t TofEventCallback;                         /*!< Callback allowing application to receive toF events */
    GENFSK_radio_config_t radioConfig;                      /*!< Copy of radioConfig used by the application, needed by ToF to determine data rate */
} gtofAppConfig_t;

/*! @brief Parameters to describe peer device.
 * @see Tof_MeasuringDeviceInit
 * @see Tof_ReceivingDeviceInit
 */
typedef struct
{
    uint32_t peerAccessAddress; /*!< Access address of peer device (Anchor or Reflector) */
    uint8_t initialChannel;     /*!< Frequency channel to be used for start exchange */
} gtofPeerConfig_t;

/*! @brief Description of measurement configuration parameters.
 * @see Tof_MeasuringDeviceStart
 */
typedef struct
{
    uint16_t nbSamples;          /*!< Number of Samples to use for  */
    uint16_t maxDurationMs;      /*!< Maximum duration of the measurement - will stop  */
    uint8_t calibIterations;     /*!< Number of measures performed during a calibration (ignored*/
    uint8_t maxRetriesPerFreq;   /*!< Number of retries allowed per frequency. Will force a channel hop if reached */
    uint8_t samplesPerFreq;      /*!< Number of samples per frequency. will force a channel hop if reached */
} gtofMeasureConfig_t;

/*! @brief Structure of Hop Log Record.
 * Contains the number of retries that happened on a given channel during the measurement.
 * @see Tof_GetFrequencyHopLog
 */
typedef struct {
    uint8_t channel;             /*!< Channel number */
    uint8_t nbRetries;           /*!< Number of packet retries that happened for the given channel */  
} ToFChannelLogEntry_t;

///////////////////////////////////////////////////////////////////////////////////////////////////
//                                Function-like Macros Section
///////////////////////////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////////////////////////
//                                  Extern Constants Section
///////////////////////////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////////////////////////
//                                  Extern Variables Section
///////////////////////////////////////////////////////////////////////////////////////////////////

/* Those variables can safely be accessed by application once TofMeasurementDoneEvent has been received */
extern uint16_t T1Measurements[];
extern uint16_t T2Measurements[];
extern uint16_t T3Measurements[];
extern uint16_t T4Measurements[];
extern fixedq6_t ToFMeasurements[];
extern uint8_t MeasuringDeviceAgcIndex[];
extern uint8_t MeasuringDevicePeerAgcIndex[];
extern fixedq6_t CalibrationValues[];
#if TOF_RSSI_LOGGING
extern int8_t MeasuringDeviceRssi[];
extern int8_t MeasuringDevicePeerRssi[];
#endif
extern int8_t MeasuringDeviceCfo[];
extern int8_t MeasuringDevicePeerCfo[];
extern const fixedq6_t AgcCompensation[];
#if TOF_CHANNEL_LOGGING
extern uint8_t MeasuringDeviceChannel[];
#endif


///////////////////////////////////////////////////////////////////////////////////////////////////
//                                Function Prototypes Section
///////////////////////////////////////////////////////////////////////////////////////////////////

#if defined(__cplusplus)
extern "C" {
#endif // __cplusplus

/*!
 * Set-up the device as a ToF MeasuringDevice (Anchor).
 * API to call before being able to use this module.
 * Prior to calling this function, the caller must have setup GENFSK config (packet, radio, crc).
 * @param [in] appConfig pointer to tofConfig containing how TOF should operate
 */
extern genfskStatus_t Tof_MeasuringDeviceInit(const gtofAppConfig_t *appConfig);

/*!
 * Set-up the device as a ToF ReceivingDevice (Reflector).
 * API to call before being able to use this module. 
 * Prior to calling this function, the caller must have setup GENFSK config (packet, radio, crc).
 * @param [in] appConfig pointer to tofConfig containing how TOF should operate
 */
extern genfskStatus_t Tof_ReceivingDeviceInit(const gtofAppConfig_t *appConfig);

/*!
 * Start calibration with given target device.
 * Will end with TofCalibrationDoneEvent or TofCalibrationError.
 * @param [in] peerConfig pointer to peer device configuration
 * @param [in] measureConfig Structure containing all measurement parameters.
 * @return TofStatusOk on success, TofStatusInvalidParameter if any input param is invalid.
 */
extern tofStatus_t Tof_MeasuringDeviceCalibrate(const gtofPeerConfig_t *peerConfig, gtofMeasureConfig_t *measureConfig);

/*!
 * Start device as a measuringDevice with given target RxDevice.
 * Will end with TofMeasurementDoneEvent or TofMeasurementErrorEvent.
 * @param [in] peerConfig pointer to peer device configuration
 * @param [in] measureConfig Structure containing all measurement parameters.
 * @return TofStatusOk on success, TofStatusInvalidParameter if any input param is invalid.
 */
extern tofStatus_t Tof_MeasuringDeviceStart(const gtofPeerConfig_t *peerConfig, gtofMeasureConfig_t *measureConfig);

/*!
 * Aborts measurement currently running.
 * Note: does not notifies Receiving device.
 */
extern void Tof_MeasuringDeviceStop(void);

/*!
 * Aborts measurement currently running and release ToF resources.
 */
extern void Tof_MeasuringDeviceRelease(void);

/*!
 * Start the device as a ReceivingDevice, using given address as local address.
 * @param [in] peerConfig pointer to peer device configuration
 */
extern void Tof_ReceivingDeviceStart(const gtofPeerConfig_t *peerConfig);

/*!
 * Aborts ReceivingDevice mode currently running.
 * Note: does not notifies Measuring device.
 */
extern void Tof_ReceivingDeviceStop(void);

/*!
 * Aborts ReceivingDevice mode currently running and release ToF resources.
 */
extern void Tof_ReceivingDeviceRelease(void);

/*!
 * @brief Set a new channel map that will be used for next ToF exchanges.
 *        By default, all channels are active.
 *        Each bit represents one channel. If the bit is set to 1, the channel is enabled
 *        If the bit is set to 0, the channel is disabled.
 *        Example: channel32_63 = 0x00000002 means that channel
 *        33 is enabled and all other channels in the range of 32 and 63 are disabled.
 *
 * @param [in] channel0_31 Configures the bits for channels 0 to 31
 * @param [in] channel32_63 Configures the bits for channels 21 to 63
 * @param [in] channel64_95 Configures the bits for channels 64 to 95
 * @param [in] channel96_127 Configures the bits for channels 96 to 127
 *
 * @return 0 if success. 1 if an invalid channel map was passed
 */
extern uint32_t ToF_GenerateNewChannelMap(uint8_t seed, uint32_t channel0_31, uint32_t channel32_63, uint32_t channel64_95, uint32_t channel96_127);

/*!
 * @brief Return pointers to Frenquency Hop log data. Both lenght and array pointer.
 *
 * @param [out] nbHops
 * @param [out] entries
 */
extern void Tof_GetFrequencyHopLog(uint32_t *nbHops, ToFChannelLogEntry_t **entries);

/*!
 * @brief Configure the device to use fixed AGC or not during measurement. If fixed AGC is chosen,
 *        AGC index is taken after first START exchange and used until the end of measurement.
 *        By default, AGC behavior is set to GTOF_AGC_BEHAVIOR_UNLOCKED.
 *
 * @param [in] agcBehavior 0=not fixed, 1=fixed, 2=fixed, using provide AGC index
 * @param [in] agcIndex Used if agcBehavior is set to GTOF_AGC_BEHAVIOR_LOCKED_INDEX
 * @param [in] maxAgcIndex will set the maximum AGCIndex
 *
 * @see gtofAgcBehavior_t
 */
extern void Tof_SetAgcBehavior(gtofAgcBehavior_t agcBehavior, uint8_t agcIndex, uint8_t maxAgcIndex);

/*!
 * @brief Configure the calibration bias directly, without running calibration process.
 *        This is useful for testing purpose, to ensure same value is used across several measures.
 *
 * @param [in] calibrationBias Value used to initialisation calibration bias
 *
 * @return 0 on success, non-zero if something wrong hapened (CalibrationBias out of bounds)
 */
extern uint32_t Tof_SetCalibrationBias(fixedq6_t calibrationBias);

/*!
 * @brief Return the current calibration bias.
 */
extern fixedq6_t Tof_GetCalibrationBias(void);

#if !defined(RADIO_IS_GEN_3P5)
/*!
 * This function must be implemented by application based on board type.
 * It must initialise MCU pins required for collecting 32MHz timestamps.
 */
extern void BOARD_InitTof(void);
#endif  /* RADIO_IS_GEN_3P5 */

#if defined(__cplusplus)
}
#endif // __cplusplus

/*! @} */

#endif /* _GTOF_H_ */
///////////////////////////////////////////////////////////////////////////////////////////////////
// EOF
///////////////////////////////////////////////////////////////////////////////////////////////////
